# Copyright 1999-2025 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2

EAPI=8

VERIFY_SIG_OPENPGP_KEY_PATH=/usr/share/openpgp-keys/isc.asc
inherit autotools flag-o-matic multiprocessing toolchain-funcs verify-sig

MY_PN=${PN//-tools}
MY_PV=${PV/_p/-P}
MY_PV=${MY_PV/_rc/rc}
MY_P="${MY_PN}-${MY_PV}"

DESCRIPTION="bind tools: dig, nslookup, host, nsupdate, dnssec-keygen"
HOMEPAGE="https://www.isc.org/software/bind https://gitlab.isc.org/isc-projects/bind9"
SRC_URI="
	https://downloads.isc.org/isc/bind9/${PV}/${MY_P}.tar.xz
	verify-sig? ( https://downloads.isc.org/isc/bind9/${PV}/${MY_P}.tar.xz.asc )
"
S="${WORKDIR}/${MY_P}"

LICENSE="Apache-2.0 BSD BSD-2 GPL-2 HPND ISC MPL-2.0"
SLOT="0"
KEYWORDS="~alpha amd64 arm arm64 ~hppa ~loong ~m68k ~mips ppc ppc64 ~riscv ~s390 ~sparc x86 ~amd64-linux ~x86-linux ~ppc-macos ~x64-macos ~x64-solaris"
IUSE="+caps doc gssapi http2 idn libedit readline xml"
# no PKCS11 currently as it requires OpenSSL to be patched, also see bug #409687

# libuv lower bound should be the highest value seen at
# https://gitlab.isc.org/isc-projects/bind9/-/blob/bind-9.16/lib/isc/netmgr/netmgr.c?ref_type=heads#L244
# to avoid issues with matching stable/testing, etc
RDEPEND="
	dev-libs/json-c
	>=dev-libs/libuv-1.42.0:=
	dev-libs/openssl:=
	caps? ( sys-libs/libcap )
	gssapi? ( virtual/krb5 )
	http2? ( net-libs/nghttp2 )
	idn? ( net-dns/libidn2:= )
	libedit? ( dev-libs/libedit )
	!libedit? (
		readline? ( sys-libs/readline:= )
	)
	xml? ( dev-libs/libxml2:= )
"
DEPEND="
	${RDEPEND}
	!>=net-dns/bind-9.18
"
# sphinx required for man-page and html creation
BDEPEND="
	dev-python/babel
	virtual/pkgconfig
	doc? ( dev-python/sphinx )
	verify-sig? ( sec-keys/openpgp-keys-isc )
"

src_prepare() {
	default

	# Do not disable thread local storage on Solaris, it works with our
	# toolchain, and it breaks further configure checks
	sed -i -e '/LDFLAGS=/s/-zrelax=transtls//' configure.ac configure || die

	# bug #220361
	rm aclocal.m4 || die
	rm -rf libtool.m4/ || die

	eautoreconf
}

src_configure() {
	local myeconfargs=(
		# localstatedir for nsupdate -l, bug #395785
		--localstatedir="${EPREFIX}"/var
		--without-zlib
		--without-lmdb
		--without-maxminddb
		--disable-geoip
		--with-openssl="${ESYSROOT}"/usr
		$(use_enable http2 doh)
		$(use_with http2 libnghttp2)
		$(use_with idn libidn2 "${ESYSROOT}"/usr)
		$(use_with xml libxml2)
		$(use_with gssapi)
		$(use_with readline)
		$(use_enable caps linux-caps)
		AR="$(type -P $(tc-getAR))"
	)

	# bug #607400
	if use libedit ; then
		myeconfargs+=( --with-readline=libedit )
	elif use readline ; then
		myeconfargs+=( --with-readline=readline )
	else
		myeconfargs+=( --without-readline )
	fi

	# bug #344029
	append-cflags "-DDIG_SIGCHASE"

	#append-ldflags "-L${ESYSROOT}/usr/$(get_libdir)"

	# to expose CMSG_* macros from sys/sockets.h
	[[ ${CHOST} == *-solaris* ]] && append-cflags "-D_XOPEN_SOURCE=600"

	tc-export BUILD_CC
	econf "${myeconfargs[@]}"

	# bug #151839
	echo '#undef SO_BSDCOMPAT' >> config.h || die
}

src_compile() {
	local AR="$(tc-getAR)"
	local dir

	emake AR="${AR}" bind.keys.h
	for dir in lib/ bin/{delv,dig,nsupdate,dnssec}/ ; do
		emake AR="${AR}" -C ${dir}
	done
	emake -C doc/man/ man $(usev doc)
}

src_test() {
	# system tests ('emake test') require network configuration for IPs etc
	# so we run the unit tests instead.
	TEST_PARALLEL_JOBS="$(makeopts_jobs)" emake -Onone unit
}

src_install() {
	local man_dir="${S}/doc/man"
	local html_dir="${man_dir}/_build/html"
	local dir

	dodoc README.md

	emake DESTDIR="${D}" -C lib/ install-exec
	for dir in bin/{delv,dig,nsupdate,dnssec}/ ; do
		emake DESTDIR="${D}" -C "${dir}" install
	done

	doman ${man_dir}/{delv,dig,host,ns{lookup,update}}.1
	if use doc ; then
		docinto html
		dodoc ${html_dir}/nsupdate.html
	fi

	local tool
	for tool in dsfromkey importkey keyfromlabel keygen revoke settime signzone verify ; do
		doman ${man_dir}/dnssec-"${tool}".1
		if use doc; then
			docinto html
			dodoc ${html_dir}/dnssec-"${tool}".html
		fi
	done

	find "${ED}" -type f -name "*.la" -delete || die
}
