# Copyright 1999-2024 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2

EAPI=8

# N.B.: It is no clue in porting to Lua eclasses, as upstream have deviated
# too far from vanilla Lua, adding their own APIs like lua_enablereadonlytable

inherit autotools edo multiprocessing systemd tmpfiles toolchain-funcs

MY_PV="${PV/_/-}"

DESCRIPTION="A persistent caching system, key-value, and data structures database"
HOMEPAGE="https://github.com/valkey-io/valkey"
SRC_URI="https://github.com/valkey-io/${PN}/archive/refs/tags/${MY_PV}.tar.gz -> ${P}.tar.gz"

LICENSE="BSD Boost-1.0"
SLOT="0/$(ver_cut 1-2)"
[[ "${PV}" == *_rc* ]] || \
KEYWORDS="~amd64 ~arm ~arm64 ~hppa ~loong ~ppc ~ppc64 ~riscv ~s390 ~sparc ~x86 ~amd64-linux ~x86-linux"
IUSE="+jemalloc selinux ssl systemd tcmalloc test"
RESTRICT="!test? ( test )"

COMMON_DEPEND="
	jemalloc? ( >=dev-libs/jemalloc-5.1:= )
	ssl? ( dev-libs/openssl:0= )
	systemd? ( sys-apps/systemd:= )
	tcmalloc? ( dev-util/google-perftools )
"

RDEPEND="
	${COMMON_DEPEND}
	acct-group/redis
	acct-user/redis
	selinux? ( sec-policy/selinux-redis )
"

BDEPEND="
	${COMMON_DEPEND}
	virtual/pkgconfig
"

# Tcl is only needed in the CHOST test env
DEPEND="
	${COMMON_DEPEND}
	test? (
		dev-lang/tcl:0=
		ssl? ( dev-tcltk/tls )
	)"

REQUIRED_USE="?? ( jemalloc tcmalloc )"

S="${WORKDIR}/${PN}-${MY_PV}"

PATCHES=(
	"${FILESDIR}"/${PN}-7.2.5-config.patch
	"${FILESDIR}"/redis-7.2.0-system-jemalloc.patch
	"${FILESDIR}"/redis-6.2.3-ppc-atomic.patch
	"${FILESDIR}"/${PN}-sentinel-7.2.5-config.patch
	"${FILESDIR}"/redis-7.0.4-no-which.patch
)

src_prepare() {
	default

	# Respect user CFLAGS in bundled lua
	sed -i '/LUA_CFLAGS/s: -O2::g' deps/Makefile || die

	# now we will rewrite present Makefiles
	local makefiles="" MKF
	local mysedconf=(
		-e 's:$(CC):@CC@:g'
		-e 's:$(CFLAGS):@AM_CFLAGS@:g'
		-e 's: $(DEBUG)::g'

		-e 's:-Werror ::g'
		-e 's:-Werror=deprecated-declarations ::g'
	)
	for MKF in $(find -name 'Makefile' | cut -b 3-); do
		mv "${MKF}" "${MKF}.in"
		sed -i "${mysedconf[@]}" "${MKF}.in" || die "Sed failed for ${MKF}"
		makefiles+=" ${MKF}"
	done
	# autodetection of compiler and settings; generates the modified Makefiles
	sed \
		-e "/^AC_INIT/s|, __PV__, |, $PV, |" \
		-e "s:AC_CONFIG_FILES(\[Makefile\]):AC_CONFIG_FILES([${makefiles}]):g" \
		"${FILESDIR}"/configure.ac-7.0 \
		> configure.ac || die "Sed failed for configure.ac"
	eautoreconf
}

src_configure() {
	econf

	# Linenoise can't be built with -std=c99, see https://bugs.gentoo.org/451164
	# also, don't define ANSI/c99 for lua twice
	sed -i -e "s:-std=c99::g" deps/linenoise/Makefile deps/Makefile || die
}

src_compile() {
	tc-export AR CC RANLIB

	local myconf=(
		AR="${AR}"
		CC="${CC}"
		RANLIB="${RANLIB}"

		V=1 # verbose

		# OPTIMIZATION defaults to -O3. Let's respect user CFLAGS by setting it
		# to empty value.
		OPTIMIZATION=''
		# Disable debug flags in bundled hiredis
		DEBUG_FLAGS=''

		BUILD_TLS=$(usex ssl)
		USE_SYSTEMD=$(usex systemd)
	)

	if use jemalloc; then
		myconf+=( MALLOC=jemalloc )
	elif use tcmalloc; then
		myconf+=( MALLOC=tcmalloc )
	else
		myconf+=( MALLOC=libc )
	fi

	emake "${myconf[@]}"
}

src_test() {
	local runtestargs=(
		--clients "$(makeopts_jobs)" # see bug #649868

		--skiptest "Active defrag eval scripts" # see bug #851654
	)

	if has usersandbox ${FEATURES} || ! has userpriv ${FEATURES}; then
		ewarn "oom-score-adj related tests will be skipped." \
			"They are known to fail with FEATURES usersandbox or -userpriv. See bug #756382."

		runtestargs+=(
			# unit/oom-score-adj was introduced in version 6.2.0
			--skipunit unit/oom-score-adj # see bug #756382

			# Following test was added in version 7.0.0 to unit/introspection.
			# It also tries to adjust OOM score.
			--skiptest "CONFIG SET rollback on apply error"
		)
	fi

	if use ssl; then
		edo ./utils/gen-test-certs.sh
		runtestargs+=( --tls )
	fi

	edo ./runtest "${runtestargs[@]}"
}

src_install() {
	local etc_dir="/etc/${PN}"
	insinto "${etc_dir}"
	doins ${PN}.conf sentinel.conf
	use prefix || fowners -R redis:redis "${etc_dir}" "${etc_dir}"/{${PN},sentinel}.conf
	fperms 0750 "${etc_dir}"
	fperms 0644 "${etc_dir}"/{${PN},sentinel}.conf

	newconfd "${FILESDIR}/valkey.confd-r2" valkey
	newinitd "${FILESDIR}/valkey.initd-6" valkey

	systemd_newunit "${FILESDIR}/valkey.service-4" valkey.service
	newtmpfiles "${FILESDIR}/valkey.tmpfiles-2" valkey.conf

	newconfd "${FILESDIR}/valkey-sentinel.confd-r1" valkey-sentinel
	newinitd "${FILESDIR}/valkey-sentinel.initd-r1" valkey-sentinel

	insinto /etc/logrotate.d/
	newins "${FILESDIR}/${PN}.logrotate" ${PN}

	dodoc 00-RELEASENOTES CONTRIBUTING.md README.md

	dobin src/${PN}-cli
	dosbin src/${PN}-{benchmark,server,check-{aof,rdb}}
	fperms 0750 /usr/sbin/${PN}-benchmark
	dosym ${PN}-server /usr/sbin/${PN}-sentinel

	if use prefix; then
		diropts -m0750
	else
		diropts -m0750 -o redis -g redis
	fi
	keepdir /var/{log,lib}/${PN}
}

pkg_postinst() {
	tmpfiles_process valkey.conf

	ewarn "The default valkey configuration file location changed to:"
	ewarn "  /etc/${PN}/{${PN},sentinel}.conf"
	ewarn "Please apply your changes to the new configuration files."
}
