# Copyright 1999-2024 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2

EAPI=7

inherit autotools db-use flag-o-matic pam systemd

DESCRIPTION="Open Source Jabber Server"
HOMEPAGE="http://jabberd2.org"
SRC_URI="https://github.com/jabberd2/jabberd2/releases/download/jabberd-${PV}/jabberd-${PV}.tar.xz"

LICENSE="GPL-2"
SLOT="0"
KEYWORDS="amd64 ~ppc ~sparc ~x86 ~x86-fbsd"
IUSE="berkdb debug experimental ldap libressl memdebug mysql pam postgres sqlite ssl test websocket zlib"
REQUIRED_USE="memdebug? ( debug )"

CDEPEND="
	dev-libs/expat
	net-libs/udns
	net-dns/libidn
	virtual/gsasl
	berkdb? ( >=sys-libs/db-4.1.25:= )
	ldap? ( net-nds/openldap )
	mysql? (
		virtual/mysql:=
		|| (
			dev-db/mariadb-connector-c
			dev-db/mysql-connector-c
		)
	)
	pam? ( sys-libs/pam )
	postgres? ( dev-db/postgresql:= )
	ssl? (
		!libressl? ( >=dev-libs/openssl-1.0.1:0=[-bindist] )
		libressl? ( dev-libs/libressl:0= )
	)
	sqlite? ( dev-db/sqlite:3 )
	websocket? ( net-libs/http-parser:= )
	zlib? ( sys-libs/zlib:= )
"
RDEPEND="
	${CDEPEND}
	>=net-im/jabber-base-0.01
"
DEPEND="
	${CDEPEND}
	test? ( dev-libs/check )
"
BDEPEND="
	app-arch/xz-utils
	dev-build/autoconf-archive
	virtual/pkgconfig
"

DOCS=( AUTHORS README )

#PATCHES=(
#)

S="${WORKDIR}/jabberd-${PV}"

pkg_pretend() {
	if is-flagq '-O[3s]' ; then
		ewarn "O3/Os compiler flags have been known to cause problems"
		ewarn "with old gcc version. Be aware that this could break"
		ewarn "port binding. Make sure to test this."
		ewarn "See https://github.com/jabberd2/jabberd2/issues/34"
	fi
}

src_prepare() {
	# Fix some default directory locations
	sed -i \
		-e 's,@localstatedir@/@package@/pid/,/var/run/@package@/,g' \
		-e 's,@localstatedir@/@package@/run/pbx,/var/run/@package@/pbx,g' \
		-e 's,@localstatedir@/@package@/log/,/var/log/@package@/,g' \
		-e 's,@localstatedir@/lib/jabberd2/fs,@localstatedir@/@package@/fs,g' \
		-e 's,@localstatedir@,/var/spool,g' \
		-e 's,@package@,jabber,g' \
		etc/{sm,router,c2s,s2s}.xml.dist.in || die "fixing default directory locations failed!"

	# If the package wasn't merged with sqlite then default to use berkdb
	use sqlite ||
		sed -i \
			-e 's,<\(module\|driver\)>sqlite<\/\1>,<\1>db</\1>,g' \
			etc/{c2s,sm}.xml.dist.in || die "setting berkdb as default failed!"

	# avoid file collision with x11-misc/screen-message wrt #453994
	sed -i \
		-e 's/@jabberd_router_bin@/jabberd2-router/' \
		-e 's/@jabberd_c2s_bin@/jabberd2-c2s/' \
		-e 's/@jabberd_s2s_bin@/jabberd2-s2s/' \
		-e 's/@jabberd_sm_bin@/jabberd2-sm/' \
		etc/jabberd*.in || die "fixing file collisions failed!"

	# rename pid files wrt #241472
	sed -i \
		-e '/pidfile/s/${id}\.pid/jabberd2-c2s\.pid/' \
		etc/c2s.xml.dist.in || die
	sed -i \
		-e '/pidfile/s/${id}\.pid/jabberd2-router\.pid/' \
		etc/router.xml.dist.in || die
	sed -i \
		-e '/pidfile/s/${id}\.pid/jabberd2-s2s\.pid/' \
		etc/s2s.xml.dist.in || die
	sed -i \
		-e '/pidfile/s/${id}\.pid/jabberd2-sm\.pid/' \
		etc/sm.xml.dist.in || die

	default

	eautoreconf
}

src_configure() {
	# --enable-pool-debug is currently broken
	myeconfargs=(
		--sysconfdir=/etc/jabber
		--enable-pipe
		--enable-anon
		--enable-fs
		$(usex debug "--enable-debug" "")
		$(usex memdebug "--enable-nad-debug" "")
		$(use_enable ssl)
		$(use_enable mysql)
		$(use_enable postgres pgsql)
		$(use_enable sqlite)
		$(use_enable berkdb db)
		$(use_enable ldap)
		$(use_enable pam)
		$(use_enable websocket)
		$(use_enable experimental)
		$(use_enable test tests)
		$(usex berkdb "--with-extra-include-path=$(db_includedir)" "") #"
		$(use_with zlib)
	)
	econf "${myeconfargs[@]}"
}

src_install() {
	# Fix systemd unit files installation path, bug #626026
	emake systemddir="$(systemd_get_systemunitdir)" DESTDIR="${D}" install
	einstalldocs
	find "${ED}" -name "*.la" -delete || die

	keepdir /var/spool/jabber/{fs,db}
	fowners jabber:jabber /usr/bin/{jabberd,router,sm,c2s,s2s} \
		/var/spool/jabber/{fs,db}
	fperms 770 /var/spool/jabber/{fs,db}
	fperms 750 /usr/bin/{jabberd,router,sm,c2s,s2s}

	# avoid file collision with x11-misc/screen-message wrt #453994
	local i
	for i in router sm c2s s2s ; do
		einfo "renaming /usr/bin/${i} to /usr/bin/jabberd2-${i}"
		mv "${ED}"/usr/bin/${i} "${ED}"/usr/bin/jabberd2-${i} || die
	done

	newinitd "${FILESDIR}/${PN}-2.5.0.init" jabberd
	newpamd "${FILESDIR}/${PN}-2.3.1.pamd" jabberd
	insinto /etc/logrotate.d
	newins "${FILESDIR}/${PN}-2.3.2.logrotate" jabberd

	docompress -x /usr/share/doc/${PF}/tools
	docinto tools
	dodoc tools/db-setup{.mysql,.pgsql,.sqlite} \
		tools/{migrate-jd14dir-2-sqlite.pl,pipe-auth.pl}

	# remove useless upstart files wrt #498900
	rm -r "${ED}"/usr/etc || die
}

pkg_postinst() {
	if use pam ; then
		echo
		ewarn 'Jabberd-2 PAM authentication requires your unix usernames to'
		ewarn 'be in the form of "contactname@jabberdomain". This behavior'
		ewarn 'is likely to change in future versions of jabberd-2. It may'
		ewarn 'be advisable to avoid PAM authentication for the time being.'
		echo
	fi

	if use sqlite || use mysql || use postgres; then
		echo
		einfo "You will need to setup or update your database using the"
		einfo "scripts in /usr/share/doc/${PF}/tools/"
		echo
	fi

	ewarn 'If you are upgrading from <=jabberd2-2.2.17 then you might have'
	ewarn 'to update /etc/jabber/jabberd.cfg via etc-update because'
	ewarn 'the binaries have been renamed to avoid file collisions!'
}
