# Copyright 1999-2025 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2
# $Id: fbd98907fad99101b9c8e76990c32720e857d7eb $

EAPI=8

MY_PV="${PV//_p/-P}"
MY_PV="${MY_PV/_/-}"
MY_P="${PN}-${MY_PV}"

PYTHON_COMPAT=( python3_{11..14} )

inherit autotools eapi9-ver flag-o-matic python-r1 systemd tmpfiles

DESCRIPTION="High-performance production grade DHCPv4 & DHCPv6 server"
HOMEPAGE="https://www.isc.org/kea/"

if [[ ${PV} == *9999* ]] ; then
	inherit git-r3
	EGIT_REPO_URI="https://gitlab.isc.org/isc-projects/kea.git"
else
	SRC_URI="https://downloads.isc.org/isc/kea/${MY_PV}/${MY_P}.tar.gz"
	# odd minor version = development release
	if [[ $(( $(ver_cut 2) % 2 )) -ne 1 ]] ; then
		if ! [[ "${PV}" == *_beta* || "${PV}" == *_rc* ]] ; then
			 KEYWORDS="amd64 ~arm64 ~x86"
		fi
	fi
fi
S="${WORKDIR}/${MY_P}"

LICENSE="MPL-2.0 GPL-2" # GPL-2 only for init script
SLOT="0"
IUSE="debug doc mysql +openssl postgres +samples shell test"

REQUIRED_USE="shell? ( ${PYTHON_REQUIRED_USE} )"
RESTRICT="!test? ( test )"

COMMON_DEPEND="
	>=dev-libs/boost-1.66:=
	dev-libs/log4cplus:=
	mysql? (
		app-arch/zstd:=
		dev-db/mysql-connector-c:=
		dev-libs/openssl:=
		virtual/zlib:=
	)
	!openssl? ( dev-libs/botan:2=[boost] )
	openssl? ( dev-libs/openssl:0= )
	postgres? ( dev-db/postgresql:* )
	shell? ( ${PYTHON_DEPS} )
"
DEPEND="${COMMON_DEPEND}
	test? ( dev-cpp/gtest )
"
RDEPEND="${COMMON_DEPEND}
	acct-group/dhcp
	acct-user/dhcp
"
BDEPEND="
	doc? (
		$(python_gen_any_dep '
			dev-python/sphinx[${PYTHON_USEDEP}]
			dev-python/sphinx-rtd-theme[${PYTHON_USEDEP}]
		')
	)
	virtual/pkgconfig
"

PATCHES=(
	"${FILESDIR}"/${PN}-2.2.0-openssl-version.patch
)

python_check_deps() {
	use doc || return 0;
	python_has_version "dev-python/sphinx[${PYTHON_USEDEP}]" \
		"dev-python/sphinx-rtd-theme[${PYTHON_USEDEP}]"
}

pkg_setup() {
	if use doc || use shell ; then
		python_setup
	fi
}

src_prepare() {
	default

	# set shebang before autotools
	if use shell; then
		sed -e 's:^#!@PYTHON@:#!/usr/bin/env python3:' \
			-i src/bin/shell/kea-shell.in || die
	fi

	# fix gtest detection
	sed -e "s:dir/lib/:dir/$(get_libdir)/:" \
		-i m4macros/ax_gtest.m4 || die

	# skip shell tests that fail to launch daemon in sandbox
	# may fail to find a suitable interface for generating a DUID-LLT
	sed -e '/TESTS += $(SHTESTS)$/d' \
		-i src/bin/dhcp4/tests/Makefile.am \
		-i src/bin/dhcp6/tests/Makefile.am || die
	sed -e '/TESTS = $(SHTESTS)$/d' \
		-i src/bin/keactrl/tests/Makefile.am || die

	# skip shell tests that require a running instance of MySQL
	if use mysql; then
		sed -e "/SHTESTS += mysql_tests.sh$/d" \
			-i src/bin/admin/tests/Makefile.am || die
	fi

	# do not create /run
	sed -e '/$(DESTDIR)${runstatedir}/d' \
		-i Makefile.am || die

	# brand the version with Gentoo
	sed -i \
		-e 's/KEA_SRCID="tarball"/KEA_SRCID="gentoo"/g' \
		-e 's/AC_MSG_RESULT("tarball")/AC_MSG_RESULT("gentoo")/g' \
		-e "s/EXTENDED_VERSION=\"\${EXTENDED_VERSION} (\$KEA_SRCID)\"/EXTENDED_VERSION=\"${PVR} (\$KEA_SRCID)\"/g" \
		configure.ac || die

	eautoreconf
}

src_configure() {
	# -Werror=odr
	# https://bugs.gentoo.org/861617
	#
	# I would truly love to submit an upstream bug but their self-hosted gitlab
	# won't let me sign up. -- Eli
	filter-lto

	local myeconfargs=(
		--disable-install-configurations
		--disable-rpath
		# manually installed
		--disable-shell
		--disable-static
		--enable-generate-messages
		--enable-logger-checks
		--enable-perfdhcp
		--localstatedir="${EPREFIX}/var"
		--runstatedir="${EPREFIX}/run"
		--without-werror
		--with-log4cplus
		$(use_enable debug)
		$(use_enable doc generate-docs)
		$(use_with mysql)
		$(use_with openssl)
		$(use_with postgres pgsql)
		$(usev test --with-gtest="${EPREFIX}/usr")
	)
	econf "${myeconfargs[@]}"
}

src_test() {
	local GTEST_SKIP_TESTS=(
		# may fail when checking addresses on detected interfaces
		IfaceMgrTest*

		# may fail to find a suitable interface for generating a DUID-LLT
		JSONFileBackendTest*
		CtrlChannelDhcpv6SrvTest*

		# require srv_config_marker_file.txt similarly generated by a disabled test
		LoadUnloadDhcpv6SrvTest*
	)

	# tests that require a running instance of MySQL
	use mysql && GTEST_SKIP_TESTS+=(
		*MySql*
		*MySQL*
		# conditional tests for MySQL
		Dhcpv*SrvTest.checkConfigFiles
	)

	# tests that require a running instance of PgSQL
	use postgres && GTEST_SKIP_TESTS+=(
		*PgSql*
		*PgSQL*
		# conditional tests for PgSQL
		Dhcpv*SrvTest.checkConfigFiles
	)

	local -x GTEST_FILTER
	[[ -n ${GTEST_SKIP_TESTS[*]} ]] && GTEST_FILTER+="-$( IFS=':'; echo "${GTEST_SKIP_TESTS[*]}")"

	default
}

install_shell() {
	python_domodule src/bin/shell/*.py
	python_doscript src/bin/shell/kea-shell

	# fix path to import kea modules
	sed -e "/^sys.path.append/s|(.*)|('$(python_get_sitedir)/${PN}')|" \
		-i "${ED}"/usr/lib/python-exec/${EPYTHON}/kea-shell || die
}

src_install() {
	emake -j1 install DESTDIR="${D}"

	if use shell; then
		python_moduleinto ${PN}
		python_foreach_impl install_shell
	fi

	dodoc -r doc/examples

	rm -f "${ED}"/usr/share/doc/${P}/COPYING

	if use samples; then
		diropts -m 0750 -o root -g dhcp
		dodir /etc/kea
		insopts -m 0640 -o root -g dhcp
		insinto /etc/kea
		doins "${FILESDIR}"/${PN}-ctrl-agent.conf
		doins "${FILESDIR}"/${PN}-ddns-server.conf
		doins "${FILESDIR}"/${PN}-dhcp4.conf
		doins "${FILESDIR}"/${PN}-dhcp6.conf
	fi

	# set log to syslog by default
	#sed -e 's/"output": "stdout"/"output": "syslog"/' \
	#	-i "${ED}"/etc/kea/*.conf.sample || die

	newconfd "${FILESDIR}"/${PN}-confd-r1 ${PN}
	newinitd "${FILESDIR}"/${PN}-initd-r3 ${PN}

	systemd_dounit "${FILESDIR}"/${PN}-ctrl-agent.service
	systemd_dounit "${FILESDIR}"/${PN}-ddns-server.service
	systemd_dounit "${FILESDIR}"/${PN}-dhcp4-server.service
	systemd_dounit "${FILESDIR}"/${PN}-dhcp6-server.service

	newtmpfiles "${FILESDIR}"/${PN}.tmpfiles.conf ${PN}.conf

	keepdir /var/lib/${PN} /var/log/${PN}
	fowners -R dhcp:dhcp /var/lib/${PN} /var/log/${PN}
	fperms 750 /var/lib/${PN} /var/log/${PN}

	find "${ED}" -type f -name "*.la" -delete || die
}

pkg_postinst() {
	tmpfiles_process ${PN}.conf
	#fcaps cap_net_bind_service,cap_net_raw=+ep usr/sbin/kea-dhcp{4,6}
}
